import { Request, Response } from 'express';
import { subscriptionService } from './service';
import { ApiResponse } from './types';

export const subscriptionsController = {
  // 1. Subscription Plans
  async getPlans(req: Request, res: Response) {
    const { data, error } = await subscriptionService.getAllPlans();
    if (error) return res.status(500).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },
  async createPlan(req: Request, res: Response) {
    const { data, error } = await subscriptionService.createPlan(req.body);
    if (error) return res.status(400).json({ success: false, error: error.message });
    return res.status(201).json({ success: true, data });
  },
  async updatePlan(req: Request, res: Response) {
    const { id } = req.params;
    const { data, error } = await subscriptionService.updatePlan(id, req.body);
    if (error) return res.status(400).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },
  async deletePlan(req: Request, res: Response) {
    const { id } = req.params;
    const { error } = await subscriptionService.deletePlan(id);
    if (error) return res.status(400).json({ success: false, error: error.message });
    return res.json({ success: true, message: 'Plan deleted' });
  },

  // 2. Vendor Subscriptions
  async getMyPlan(req: any, res: Response) {
    const vendorId = req.user?.id || req.vendor?.id;
    if (!vendorId) return res.status(401).json({ success: false, error: 'Unauthorized' });
    const { data, error } = await subscriptionService.getVendorSubscription(vendorId);
    if (error) return res.status(500).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },
  async activatePlan(req: any, res: Response) {
    const vendorId = req.user?.id || req.vendor?.id;
    const { plan_id } = req.body;
    if (!vendorId || !plan_id) return res.status(400).json({ success: false, error: 'Missing vendor or plan' });
    const { data, error } = await subscriptionService.activateSubscription(vendorId, plan_id);
    if (error) return res.status(400).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },
  async getTrialOffer(req: any, res: Response) {
    const { data, error } = await subscriptionService.getTrialSettings();
    if (error) return res.status(500).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },
  async acceptTrial(req: any, res: Response) {
    const vendorId = req.user?.id || req.vendor?.id;
    const { plan_id } = req.body;
    const { data: trialSettings } = await subscriptionService.getTrialSettings();
    if (!trialSettings?.is_enabled) return res.status(400).json({ success: false, error: 'Trial not available' });
    const { data, error } = await subscriptionService.acceptTrial(vendorId, plan_id, trialSettings.duration_days);
    if (error) return res.status(400).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },

  // 3. Features (admin)
  async getAllFeatures(req: Request, res: Response) {
    const { data, error } = await subscriptionService.getAllFeatures();
    if (error) return res.status(500).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },
  async updateFeature(req: Request, res: Response) {
    const { id } = req.params;
    const { data, error } = await subscriptionService.updateFeature(id, req.body);
    if (error) return res.status(400).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },
  async assignFeatureToPlan(req: Request, res: Response) {
    const { plan_id, feature_id } = req.body;
    const { data, error } = await subscriptionService.assignFeatureToPlan(plan_id, feature_id);
    if (error) return res.status(400).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },
  async removeFeatureFromPlan(req: Request, res: Response) {
    const { plan_id, feature_id } = req.body;
    const { error } = await subscriptionService.removeFeatureFromPlan(plan_id, feature_id);
    if (error) return res.status(400).json({ success: false, error: error.message });
    return res.json({ success: true, message: 'Feature removed from plan' });
  },

  // 4. Trial Settings (admin)
  async updateTrialSettings(req: Request, res: Response) {
    const { data, error } = await subscriptionService.updateTrialSettings(req.body);
    if (error) return res.status(400).json({ success: false, error: error.message });
    return res.json({ success: true, data });
  },

  // 5. Feature Gate: get available features for a user
  async getAvailableFeatures(req: any, res: Response) {
    const userId = req.user?.id;
    if (!userId) return res.status(401).json({ success: false, error: 'Unauthorized' });
    const features = await subscriptionService.getAvailableFeatures(userId);
    return res.json({ success: true, data: features });
  }
}; 